package net.robertboehm.ld24.scenes;

import haxe.FastList;
import net.robertboehm.ld24.entities.buildings.Building;
import net.robertboehm.ld24.entities.buildings.CrystalExtractor;
import net.robertboehm.ld24.entities.buildings.CrystalMine;
import net.robertboehm.ld24.entities.buildings.GunTurret;
import net.robertboehm.ld24.entities.buildings.RapidTurret;
import net.robertboehm.ld24.entities.buildings.SellBuilding;
import net.robertboehm.ld24.entities.buildings.SniperTurret;
import net.robertboehm.ld24.entities.buildings.Turret;
import net.robertboehm.ld24.entities.CoolList;
import net.robertboehm.ld24.entities.enemies.BigEnemy;
import net.robertboehm.ld24.entities.enemies.Enemy;
import net.robertboehm.ld24.entities.enemies.NormalEnemy;
import net.robertboehm.ld24.entities.enemies.TinyEnemy;
import net.robertboehm.ld24.entities.EnemyList;
import net.robertboehm.ld24.entities.Gui;
import net.robertboehm.ld24.entities.Map;
import net.robertboehm.ld24.entities.Projectile;
import net.robertboehm.ld24.util.Enhancements;
import net.robertboehm.ld24.util.GameAssets;
import net.robertboehm.ld24.util.Main;
import net.robertboehm.ld24.util.Tile;
import org.rygal.Game;
import org.rygal.GameTime;
import org.rygal.graphic.Canvas;
import org.rygal.graphic.Font;
import org.rygal.graphic.Texture;
import org.rygal.input.Controller;
import org.rygal.input.DirectionalInput;
import org.rygal.input.KeyboardEvent;
import org.rygal.input.Keys;
import org.rygal.input.Keyset;
import org.rygal.input.MouseEvent;
import org.rygal.Scene;
import org.rygal.ui.Label;
import org.rygal.util.Storage;

/**
 * ...
 * @author Robert Böhm
 */

class GameScene extends Scene {
	
	static public inline var CAMERA_SPEED:Float = 3000;
	static public inline var CAMERA_FRICTION:Float = 0.89;
	static public inline var CAMERA_MOVEMENT_PADDING:Float = 8;
	static public inline var CAMERA_MOUSE_SPEED:Float = 2000;
	
	private var speedLabel:Label;
	
	public var enemies:EnemyList;
	public var projectiles:CoolList<Projectile>;
	public var crystals(getCrystals, setCrystals):Int;
	public var wannaBuild:Building;
	public var timeLeft:Float;
	public var startDelay:Float;
	public var enhancements:Enhancements;
	public var mapname(default, null):String;
	
	private var _speed:Bool;
	private var _buildingList:FastList<Building>;
	private var _crystals:Int = 20;
	private var map:Map;
	private var buildings:Array<Building>;
	private var camVelocityX:Float = 0;
	private var camVelocityY:Float = 0;
	private var controller:Controller;
	private var gui:Gui;
	private var nextMap:String;
	private var _speedMessageWait:Float = 0;
	public var built(default, null):Bool = false;
	
	
	public function new(mapname:String, ?nextMap:String) {
		super();
		
		this.mapname = mapname;
		this.nextMap = nextMap;
		
		enhancements = new Enhancements();
		
		this.map = new Map(mapname);
		this.map.enhancements = enhancements;
		this.projectiles = new CoolList<Projectile>();
		this.enemies = new EnemyList();
		_speed = new Storage("savegame").get("speed", false);
	}
	
	
	override public function load(game:Game):Void {
		super.load(game);
		
		enhancements.restore(new Storage("savegame"));
		
		speedLabel = new Label(GameAssets.font, "Press [SPACE] for 3x speed", Font.RIGHT, 0, 1, game.width - 4, Gui.TOP_BAR + 4);
		this.built = false;
		this.wannaBuild = null;
		this._crystals = map.startCrystals;
		
		this.addChild(map);
		this.addChild(projectiles);
		
		this.startDelay = map.startDelay;
		this.timeLeft = map.time;
		this._buildingList = new FastList<Building>();
		this.buildings = new Array<Building>();
		this.buildings[map.width * map.height - 1] = null;
		
		for (x in 0...map.width) {
			for (y in 0...map.height) {
				if (map.getTile(x, y) == Tile.BIG_CRYSTAL) {
					var ce:CrystalExtractor = new CrystalExtractor(this,
						x * GameAssets.spritesheet.spriteWidth, y * GameAssets.spritesheet.spriteHeight);
					this.addChild(this.buildings[x + y * map.width] = ce);
					this._buildingList.add(ce);
				}
			}
		}
		
		this.addChild(enemies);
		
		this.game.mouse.addEventListener(MouseEvent.MOUSE_DOWN, onMouseDown);
		this.game.keyboard.addEventListener(KeyboardEvent.KEY_DOWN, onKeyDown);
		
		this.controller = new Controller(game);
		this.controller.registerInput("cameraMovement", DirectionalInput);
		this.controller.bindKeyset("cameraMovement", Keyset.getWASD());
		this.controller.bindKeyset("cameraMovement", Keyset.getArrowKeys());
		this.controller.bindKeyset("cameraMovement", Keyset.getIJKL());
		
		this.addChild(controller);
		
		this.gui = new Gui(this);
	}
	
	override public function unload():Void {
		super.unload();
		
		this.game.cameraX = 0;
		this.game.cameraY = 0;
		
		this.game.mouse.removeEventListener(MouseEvent.MOUSE_DOWN, onMouseDown);
		this.game.keyboard.removeEventListener(KeyboardEvent.KEY_DOWN, onKeyDown);
		
		this.enemies.clear();
		this.projectiles.clear();
	}
	
	override public function update(time:GameTime):Void {
		Main.pinkiePie.update(time);
		
		_speedMessageWait += time.elapsedS;
		
		camVelocityX += controller.getDirectionalVectorX("cameraMovement") * CAMERA_SPEED * time.elapsedS;
		camVelocityY += controller.getDirectionalVectorY("cameraMovement") * CAMERA_SPEED * time.elapsedS;
		
		game.cameraX += (camVelocityX *= CAMERA_FRICTION) * time.elapsedS;
		game.cameraY += (camVelocityY *= CAMERA_FRICTION) * time.elapsedS;
		
		var screenMouseX:Float = game.mouse.x - game.cameraX;
		var screenMouseY:Float = game.mouse.y - game.cameraY;
		
		if (screenMouseX <= CAMERA_MOVEMENT_PADDING) camVelocityX -= CAMERA_MOUSE_SPEED * time.elapsedS;
		if (screenMouseY <= CAMERA_MOVEMENT_PADDING) camVelocityY -= CAMERA_MOUSE_SPEED * time.elapsedS;
		if (screenMouseX >= game.width - CAMERA_MOVEMENT_PADDING) camVelocityX += CAMERA_MOUSE_SPEED * time.elapsedS;
		if (screenMouseY >= game.height - Gui.TOP_BAR - CAMERA_MOVEMENT_PADDING) camVelocityY += CAMERA_MOUSE_SPEED * time.elapsedS;
		
		if (game.cameraX < 0) game.cameraX = 0;
		if (game.cameraY < 0) game.cameraY = 0;
		if (game.cameraX + game.width > map.getPixelWidth()) game.cameraX = map.getPixelWidth() - game.width;
		if (game.cameraY + game.height - Gui.TOP_BAR > map.getPixelHeight()) game.cameraY = map.getPixelHeight() - game.height + Gui.TOP_BAR;
		
		if (game.keyboard.isKeyPressed(Keys.SPACE)) {
			speedLabel.text = "3x speed";
			time.applyTimeFactor(3);
			if (!_speed) {
				new Storage("savegame").put("speed", true);
				_speed = true;
			}
		} else {
			if (_speed || _speedMessageWait < 10) {
				speedLabel.text = "";
			} else {
				speedLabel.text = "Press [SPACE] for 3x speed";
			}
		}
		
		GameAssets.playShot = true;
		GameAssets.playAttack = true;
		GameAssets.playKill = true;
		GameAssets.playMoney = true;
		
		super.update(time);
		
		if (this.startDelay > 0) {
			if (this.built) {
				startDelay -= time.elapsedS;
				if (startDelay <= 0) {
					this.timeLeft += startDelay;
				}
			}
		} else {
			this.timeLeft -= time.elapsedS;
			if (this.timeLeft <= 0) {
				this.timeLeft = 0;
				if (this.nextMap == null) {
					game.useScene("gamewin");
				} else {
					var storage:Storage = new Storage("savegame");
					storage.put("nextLevel", this.nextMap);
					game.useScene("enhancements");
				}
			}
		}
		
		if (this.startDelay <= 0) {
			for (x in 0...map.width) {
				for (y in 0...map.height) {
					if (map.getTile(x, y) == Tile.ENEMY_SPAWN) {
						var i:Int = x + y * map.width;
						var progress:Float = timeLeft / map.time;
						var minRate:Float = map.minSpawnRate * progress + map.endMinSpawnRate * (1 - progress);
						var maxRate:Float = map.maxSpawnRate * progress + map.endMaxSpawnRate * (1 - progress);
						var spawnRate:Float = Math.random() * (maxRate - minRate) + minRate;
						map.spawnCooldowns[i] -= time.elapsedS / spawnRate;
						while (map.spawnCooldowns[i] < 0) {
							map.spawnCooldowns[i] += spawnRate;
							
							spawn(x * GameAssets.spritesheet.spriteWidth + GameAssets.spritesheet.spriteWidth / 2,
								y * GameAssets.spritesheet.spriteHeight + GameAssets.spritesheet.spriteHeight / 2);
						}
					}
				}
			}
		}
		
		
		var previous:FastCell<Building> = null;
		var current:FastCell<Building> = _buildingList.head;
		
		while (current != null) {
			if (current.elt.isDead()) {
				var distanceFactor:Float = this.getDistanceVolumeFactor(current.elt.x + current.elt.width / 2,
					current.elt.y + current.elt.height / 2);
				GameAssets.explosionSound.play(GameAssets.soundVolume * distanceFactor);
				
				if (Std.is(current.elt, CrystalExtractor)) {
					game.useScene("gameover");
				}
				removeChild(current.elt);
				for (i in 0...buildings.length) {
					if (buildings[i] == current.elt) {
						buildings[i] = null;
					}
				}
				
				if (previous == null) {
					if (_buildingList.head == current) {
						// Still the first element
						_buildingList.head = current.next;
					} else {
						// Another element was added
						previous = _buildingList.head;
						while (previous.next != null && previous.next != current) {
							previous = previous.next;
						}
						previous.next = current.next;
					}
				} else {
					previous.next = current.next;
				}
			}
			previous = current;
			current = current.next;
		}
		
		this.gui.update(time);
	}
	
	override public function draw(screen:Canvas):Void {
		screen.push();
		screen.translate(0, Gui.TOP_BAR);
		super.draw(screen);
		
		for (x in 0...map.width) {
			for (y in 0...map.height) {
				if (map.getTile(x, y) == Tile.ENEMY_SPAWN) {
					screen.draw(GameAssets.spritesheet.getTexture(2, 3), x * GameAssets.spritesheet.spriteWidth,
						y * GameAssets.spritesheet.spriteHeight);
				}
			}
		}
		
		if (wannaBuild != null && !gui.isHovering(game.mouse.x - game.cameraX, game.mouse.y - game.cameraY)) {
			var tx:Int = getMouseTileX();
			var ty:Int = getMouseTileY();
			if (canBuild(tx, ty)) {
				screen.draw(GameAssets.spritesheet.getTexture(3, 4), tx * GameAssets.spritesheet.spriteWidth,
					ty * GameAssets.spritesheet.spriteHeight);
				var txt:Texture = null;
				if (Std.is(wannaBuild, GunTurret)) {
					txt = GameAssets.transparentSpritesheet.getTexture(0, 2);
				} else if (Std.is(wannaBuild, RapidTurret)) {
					txt = GameAssets.transparentSpritesheet.getTexture(1, 2);
				} else if (Std.is(wannaBuild, SniperTurret)) {
					txt = GameAssets.transparentSpritesheet.getTexture(2, 2);
				} else if (Std.is(wannaBuild, CrystalMine)) {
					txt = GameAssets.transparentSpritesheet.getTexture(3, 0);
				}
				screen.draw(txt, tx * GameAssets.spritesheet.spriteWidth, ty * GameAssets.spritesheet.spriteHeight);
			} else {
				screen.draw(GameAssets.spritesheet.getTexture(3, 5), tx * GameAssets.spritesheet.spriteWidth,
					ty * GameAssets.spritesheet.spriteHeight);
			}
		}
		
		screen.pop();
		
		this.gui.draw(screen);
		
		screen.push();
		screen.translate( -screen.xTranslation, -screen.yTranslation);
		
		speedLabel.draw(screen);
		
		screen.pop();
		
		Main.pinkiePie.draw(screen);
	}
	
	public function getClosestBuildingsOfType<T:Building>(type:Class<T>, source:Enemy, amount:Int):Array<T> {
		var list:Array<T> = new Array<T>();
		for (i in 0...amount) {
			var minDistance:Float = 0;
			var minDistanceBuilding:T = null;
			for (building in _buildingList) {
				if (!Std.is(building, type) || building.isDead())
					continue;
				
				var contained:Bool = false;
				for (obj in list) {
					if (obj == building) {
						contained = true;
						break;
					}
				}
				var distance:Float = source.getDistanceTo(building.x + building.width / 2, building.y + building.height / 2);
				if (minDistanceBuilding == null || minDistance > distance) {
					minDistance = distance;
					minDistanceBuilding = cast building;
				}
			}
			if (minDistanceBuilding != null) {
				list.push(minDistanceBuilding);
			} else {
				break;
			}
		}
		return list;
	}
	
	public function getClosestBuildingOfType<T:Building>(type:Class<T>, source:Enemy):T {
		var buildings:Array<T> = getClosestBuildingsOfType(type, source, 1);
		if (buildings.length > 0) {
			return buildings[0];
		} else {
			return null;
		}
	}
	
	public function getDistanceVolumeFactor(x:Float, y:Float):Float {
		var centerX:Float = game.cameraX + game.width / 2;
		var centerY:Float = game.cameraY + game.height / 2;
		var diffX:Float = centerX - x;
		var diffY:Float = centerY - y;
		diffX = Math.max(0, Math.abs(diffX) - game.width / 2);
		diffY = Math.max(0, Math.abs(diffY) - game.height / 2);
		if (diffX == 0 && diffY == 0) {
			return 1;
		}
		var distance:Float = Math.sqrt(diffX * diffX + diffY * diffY);
		return Math.min(1, 1 / Math.sqrt(Math.sqrt(distance)));
	}
	
	
	private function spawn(x:Float, y:Float):Void {
		var progress:Float = 1 - (timeLeft / map.time);
		var modifier:Float = map.monsterProgressBase + progress * map.monsterProgressFactor;
		var count:Int = (Math.random() <= map.doubleChance) ? 2 : 1;
		for (i in 0...count) {
			var rnd:Float = Math.random();
			if (rnd <= 0.025) {
				enemies.spawn(new BigEnemy(this, modifier, x, y));
			} else if (rnd <= 0.3) {
				enemies.spawn(new NormalEnemy(this, modifier, x, y));
			} else {
				enemies.spawn(new TinyEnemy(this, modifier, x, y));
			}
		}
	}
	
	private function onMouseDown(e:MouseEvent):Void {
		if (!game.isPaused() && wannaBuild != null && !gui.isHovering(game.mouse.x - game.cameraX, game.mouse.y - game.cameraY)) {
			
			var tx:Int = getMouseTileX();
			var ty:Int = getMouseTileY();
			
			if (wannaBuild.getPrice() <= this.crystals && canBuild(tx, ty)) {
				var index:Int = tx + ty * this.map.width;
				var building:Building = buildings[index];
				if (building != null) {
					removeChild(building);
					_buildingList.remove(building);
				}
				
				if (Std.is(wannaBuild, SellBuilding)) {
					this.crystals += Math.round(buildings[index].getPriceFactor() *
						buildings[index].getPrice() * Building.SELL_FACTOR);
					
					buildings[index] = null;
					
					GameAssets.explosionSound.play(GameAssets.soundVolume);
					
				} else {
					this.built = true;
					
					var bx:Int = tx * GameAssets.spritesheet.spriteWidth;
					var by:Int = ty * GameAssets.spritesheet.spriteHeight;
					
					this.crystals -= wannaBuild.getPrice();
					buildings[index] = wannaBuild;
					buildings[index].x = bx;
					buildings[index].y = by;
					
					_buildingList.add(buildings[index]);
					
					addChildAt(buildings[index], 2);
					
					GameAssets.buildingSound.play(GameAssets.soundVolume);
				}
				
				if (game.keyboard.isKeyPressed(Keys.SHIFT)) {
					wannaBuild = wannaBuild.clone();
				} else {
					wannaBuild = null;
				}
			}
		}
	}
	
	
	private function getCrystals():Int {
		return _crystals;
	}
	
	private function setCrystals(crystals:Int):Int {
		var difference:Int = crystals - _crystals;
		if (difference != 0) {
			gui.addCrystalChange(difference);
		}
		/*
		 * No money sounds
		if (difference > 0 && GameAssets.playMoney) {
			GameAssets.moneySound.play(GameAssets.soundVolume);
			GameAssets.playMoney = false;
		}*/
		return _crystals = crystals;
	}
	
	private function onKeyDown(e:KeyboardEvent):Void {
		if (e.keyCode == Keys.P) {
			game.pause();
		} else if (e.keyCode == Keys.ESCAPE) {
			wannaBuild = null;
		} else if (e.keyCode == Keys.M) {
			Main.toggleSound();
		}
	}
	
	private function canBuild(x:Int, y:Int):Bool {
		if (Std.is(wannaBuild, Turret)) {
			return (map.getTile(x, y) == Tile.EMPTY || (map.getTile(x, y) == Tile.EXPLORER_CRYSTAL && !enhancements.explorer))
				&& buildings[x + y * map.width] == null;
		} else if (Std.is(wannaBuild, CrystalMine)) {
			return (map.getTile(x, y) == Tile.SMALL_CRYSTAL ||
				(map.getTile(x, y) == Tile.EXPLORER_CRYSTAL && enhancements.explorer)) &&
				buildings[x + y * map.width] == null;
		} else if (Std.is(wannaBuild, SellBuilding)) {
			return buildings[x + y * map.width] != null && !Std.is(buildings[x + y * map.width], CrystalExtractor);
		}
		return false;
	}
	
	private function getMouseTileX():Int {
		return Std.int(Math.max(0, Math.min(map.width - 1, game.mouse.x / GameAssets.spritesheet.spriteWidth)));
	}
	
	private function getMouseTileY():Int {
		return Std.int(Math.max(0, Math.min(map.height - 1, (game.mouse.y - Gui.TOP_BAR) / GameAssets.spritesheet.spriteHeight)));
	}
	
}
