package net.robertboehm.ld24.entities.enemies;

import net.robertboehm.ld24.entities.buildings.Building;
import net.robertboehm.ld24.scenes.GameScene;
import net.robertboehm.ld24.util.EnemyAI;
import net.robertboehm.ld24.util.GameAssets;
import net.robertboehm.ld24.util.Livable;
import org.rygal.GameTime;
import org.rygal.graphic.AnimatedSprite;

/**
 * ...
 * @author Robert Böhm
 */

class Enemy extends AnimatedSprite, implements Livable {
	
	public var movementDirection:Float = 0;
	public var movement:Float = 0;
	public var attackRate:Float;
	public var damage:Float;
	public var speed:Float;
	public var health:Float;
	public var ai:EnemyAI;
	public var scene:GameScene;
	private var _attackCooldown:Float = 0;
	
	
	public function new(scene:GameScene, x:Float = 0, y:Float = 0) {
		super(x, y);
		
		this.scene = scene;
	}
	
	
	public function getX():Float {
		return this.x + this.width / 2;
	}
	
	public function getY():Float {
		return this.y + this.height / 2;
	}
	
	public function isDead():Bool {
		return this.health <= 0 && this.getCurrentAnimation() != "die";
	}
	
	private var diffX:Float;
	private var diffY:Float;
	
	public function getSquaredDistanceTo(x:Float, y:Float):Float {
		diffX = getX() - x;
		diffY = getY() - y;
		return diffX * diffX + diffY * diffY;
	}
	
	public function getDistanceTo(x:Float, y:Float):Float {
		diffX = getX() - x;
		diffY = getY() - y;
		return Math.sqrt(diffX * diffX + diffY * diffY);
	}
	
	public function getDirectionTo(x:Float, y:Float):Float {
		return Math.atan2(y - getY(), x - getX());
	}
	
	override public function update(time:GameTime):Void {
		if (ai != null)
			ai.think(this);
		
		super.update(time);
		
		if (this.health > 0 && movement > 0) {
			var currentMovement:Float = Math.min(time.elapsedS * speed, movement);
			movement -= currentMovement;
			
			var xMovement:Float = Math.cos(movementDirection) * currentMovement;
			var yMovement:Float = Math.sin(movementDirection) * currentMovement;
			if (Math.abs(xMovement) > Math.abs(yMovement)) {
				if (xMovement > 0) {
					this.loop("moveRight");
				} else {
					this.loop("moveLeft");
				}
			} else {
				if (yMovement > 0) {
					this.loop("moveDown");
				} else {
					this.loop("moveUp");
				}
			}
			this.x += xMovement;
			this.y += yMovement;
		} else {
			if (this.getCurrentAnimation() == "moveRight") {
				this.loop("idleRight");
			} else if (this.getCurrentAnimation() == "moveLeft") {
				this.loop("idleLeft");
			} else if (this.getCurrentAnimation() == "moveDown") {
				this.loop("idleDown");
			} else if (this.getCurrentAnimation() == "moveUp") {
				this.loop("idleUp");
			}
		}
		
		this._attackCooldown += time.elapsedMs;
		while (this._attackCooldown >= this.attackRate) {
			var closest:Building = scene.getClosestBuildingOfType(Building, this);
			if (closest != null) {
				var distance:Float = this.getDistanceTo(closest.x + closest.width / 2, closest.y + closest.height / 2);
				if (distance <= 10) {
					closest.hit(this.damage);
					this._attackCooldown -= this.attackRate;
				} else {
					this._attackCooldown = this.attackRate;
					break;
				}
			} else {
				this._attackCooldown = this.attackRate;
				break;
			}
		}
		
	}
	
	public function hit(damage:Float) {
		this.health -= damage;
		if (this.health <= 0) {
			this.play("die");
			if (GameAssets.playKill) {
				var distanceFactor:Float = scene.getDistanceVolumeFactor(this.getX(), this.getY());
				if (distanceFactor > GameAssets.MIN_DISTANCEFACTOR) {
					GameAssets.killSound.play(GameAssets.soundVolume * distanceFactor);
					GameAssets.playKill = false;
				}
			}
		}
	}
}