package net.robertboehm.ld24.entities;

import net.robertboehm.ld24.entities.buildings.Building;
import net.robertboehm.ld24.entities.buildings.CrystalMine;
import net.robertboehm.ld24.entities.buildings.GunTurret;
import net.robertboehm.ld24.entities.buildings.RapidTurret;
import net.robertboehm.ld24.entities.buildings.SellBuilding;
import net.robertboehm.ld24.entities.buildings.SniperTurret;
import net.robertboehm.ld24.scenes.GameScene;
import net.robertboehm.ld24.util.GameAssets;
import org.rygal.BasicGameObject;
import org.rygal.GameTime;
import org.rygal.graphic.Canvas;
import org.rygal.graphic.Color;
import org.rygal.graphic.Font;
import org.rygal.graphic.Texture;
import org.rygal.physics.Rectangle;
import org.rygal.ui.Label;

/**
 * ...
 * @author Robert Böhm
 */

class Gui extends BasicGameObject {
	
	public static inline var TOP_BAR:Int = 13;
	public static inline var HOVER_DELAY:Float = 0.5;
	public static inline var BUILDINGS:Int = 4;
	public static inline var CHANGESPEED:Float = 20;
	
	public var padding:Int = 8;
	
	public var scene(default, null):GameScene;
	private var crystalAmount:Label;
	private var greenCrystalAmount:Label;
	private var abortBuilding:Label;
	private var toggleSound:Label;
	private var buildingName:Label;
	private var hovered:Float = 0;
	private var changes:Array<Label>;
	
	
	public function new(scene:GameScene) {
		super();
		
		this.changes = new Array<Label>();
		this.scene = scene;
		this.crystalAmount = new Label(GameAssets.font, Std.string(scene.crystals), Font.LEFT, 0xC8DEEC, 1, 9, 0);
		this.greenCrystalAmount = new Label(GameAssets.font, "", Font.RIGHT, 0xCCECC8, 1, scene.game.width - 8, 0);
		
		this.toggleSound = new Label(GameAssets.font, "[M] Mute sound", Font.CENTER, Color.LIGHT_GRAY, 1, scene.game.width / 2, 0);
		this.abortBuilding = new Label(GameAssets.font, "[ESC] Abort building", Font.CENTER, Color.LIGHT_GRAY, 1, scene.game.width / 2, 0);
		this.buildingName = new Label(GameAssets.font, "Please select...", Font.LEFT, Color.LIGHT_GRAY, 1, padding + 6, scene.game.height - padding - GameAssets.spritesheet.spriteHeight * 2 + 20);
	}
	
	
	public function isHovering(x:Float, y:Float):Bool {
		var sw:Float = GameAssets.spritesheet.spriteWidth;
		var sh:Float = GameAssets.spritesheet.spriteHeight;
		
		var rectangle:Rectangle;
		if (hovered > 0) {
			rectangle = new Rectangle(padding, scene.game.height - sh - padding, sw * (BUILDINGS + 1), sh);
		} else {
			rectangle = new Rectangle(padding, scene.game.height - sh - padding, sw, sh);
		}
		return rectangle.contains(x, y);
	}
	
	public function addCrystalChange(change:Int):Void {
		if (change == 0)
			return;
		
		if (changes.length > 1) {
			var newChange:Int = change + Std.parseInt(changes[1].text);
			if (newChange == 0) {
				changes.splice(1, 1);
			} else {
				changes[1].text = (newChange >= 0 ? "+" : "") + Std.string(newChange);
				// Refresh x-coordinate:
				changes[1].x = crystalAmount.x + crystalAmount.font.getTextMetrics(crystalAmount.text).width + 8;
			}
		} else {
			changes.push(new Label(GameAssets.font, (change >= 0 ? "+" : "") + Std.string(change), Font.LEFT, 0xC8DEEC,
				1, crystalAmount.x + crystalAmount.font.getTextMetrics(crystalAmount.text).width + 4, Gui.TOP_BAR));
		}
	}
	
	override public function update(time:GameTime):Void {
		var sw:Float = GameAssets.spritesheet.spriteWidth;
		
		super.update(time);
		
		if (scene.startDelay > 0) {
			if (scene.built) {
				greenCrystalAmount.text = "Prepare (" + Math.round(scene.startDelay) + " seconds left)";
			} else {
				greenCrystalAmount.text = "Prepare";
			}
		} else {
			var timeLeftSeconds:Int = Math.round(scene.timeLeft);
			if (timeLeftSeconds > 60) {
				greenCrystalAmount.text = Math.ceil(timeLeftSeconds / 60) + " minutes left";
			} else if (timeLeftSeconds <= 0) {
				greenCrystalAmount.text = "All crystals collected";
			} else {
				greenCrystalAmount.text = timeLeftSeconds + " seconds left";
			}
		}
		
		if (changes.length > 0) {
			changes[0].y -= time.elapsedS * CHANGESPEED;
			changes[0].update(time);
			
			if (changes[0].y + changes[0].font.getTextMetrics(changes[0].text).height < 0) {
				changes.shift();
			}
		}
		
		if (isHovering(scene.game.mouse.x - scene.game.cameraX, scene.game.mouse.y - scene.game.cameraY)) {
			hovered = HOVER_DELAY;
			
			var i:Int = Std.int((scene.game.mouse.x - scene.game.cameraX - padding) / sw);
			if (scene.game.mouse.isPressed) {
				scene.wannaBuild = switch(i) {
					case 0: new SellBuilding();
					case 1: new CrystalMine(scene);
					case 2: new GunTurret(scene);
					case 3: new RapidTurret(scene);
					case 4: new SniperTurret(scene);
					default: null;
				}
			} else {
				buildingName.text = switch(i) {
					case 0: "Sell";
					case 1: CrystalMine.NAME + " (" + CrystalMine.PRICE + "$)";
					case 2: GunTurret.NAME + " (" + GunTurret.PRICE + "$)";
					case 3: RapidTurret.NAME + " (" + RapidTurret.PRICE + "$)";
					case 4: SniperTurret.NAME + " (" + SniperTurret.PRICE + "$)";
					default: "Please select...";
				}
			}
		} else {
			hovered -= time.elapsedS;
			buildingName.text = "Please select...";
		}
	}
	
	override public function draw(screen:Canvas):Void {
		var sw:Float = GameAssets.spritesheet.spriteWidth;
		var sh:Float = GameAssets.spritesheet.spriteHeight;
		
		screen.push();
		screen.translate( -screen.xTranslation, -screen.yTranslation);
		
		super.draw(screen);
		
		var hudLeft:Texture = GameAssets.hud.slice(0, 0, 32, GameAssets.hud.height);
		var hudMiddle:Texture = GameAssets.hud.slice(32, 0, 64, GameAssets.hud.height);
		var hudRight:Texture = GameAssets.hud.slice(GameAssets.hud.width - 32, 0, 32, GameAssets.hud.height);
		for (i in 0...Math.ceil(screen.width / 64)) {
			screen.draw(hudMiddle, i * 64, 0);
		}
		screen.draw(hudLeft, 0, 0);
		screen.draw(hudRight, screen.width - 32, 0);
		
		if (changes.length > 0) {
			var changeY:Float = changes[0].y;
			changes[0].y = Math.round(changeY);
			changes[0].drawClipped(screen, new Rectangle(0, 1, screen.width, TOP_BAR - 2));
			changes[0].y = changeY;
		}
		
		crystalAmount.text = Std.string(scene.crystals);
		
		crystalAmount.draw(screen);
		greenCrystalAmount.draw(screen);
		
		if (scene.wannaBuild != null) {
			abortBuilding.draw(screen);
		} else {
			toggleSound.draw(screen);
		}
		if (hovered <= 0) {
			screen.draw(GameAssets.spritesheet.getTexture(0, 3), padding, scene.game.height - sh - padding);
		} else {
			
			screen.draw(GameAssets.spritesheet.getTexture(0, 4), padding + sw * 0, scene.game.height - sh * 2 - padding);
			screen.draw(GameAssets.spritesheet.getTexture(0, 5), padding + sw * 0, scene.game.height - sh - padding);
			for (i in 1...BUILDINGS) {
				screen.draw(GameAssets.spritesheet.getTexture(1, 4), padding + sw * i, scene.game.height - sh * 2 - padding);
				screen.draw(GameAssets.spritesheet.getTexture(1, 5), padding + sw * i, scene.game.height - sh - padding);
			}
			screen.draw(GameAssets.spritesheet.getTexture(2, 4), padding + sw * BUILDINGS, scene.game.height - sh * 2 - padding);
			screen.draw(GameAssets.spritesheet.getTexture(2, 5), padding + sw * BUILDINGS, scene.game.height - sh - padding);
			
			screen.draw(GameAssets.spritesheet.getTexture(1, 3), padding + sw * 0, scene.game.height - sh - padding);
			screen.draw(GameAssets.spritesheet.getTexture(3, 0), padding + sw * 1, scene.game.height - sh - padding);
			screen.draw(GameAssets.spritesheet.getTexture(0, 2), padding + sw * 2, scene.game.height - sh - padding);
			screen.draw(GameAssets.spritesheet.getTexture(1, 2), padding + sw * 3, scene.game.height - sh - padding);
			screen.draw(GameAssets.spritesheet.getTexture(2, 2), padding + sw * 4, scene.game.height - sh - padding);
			
			buildingName.draw(screen);
		}
		
		screen.pop();
	}
	
}